/*Based on ideas from the following sites:
 
  Example program for timer with an ISR example
  Author: Nick Gammon, http://www.gammon.com.au/interrupts (Ignition timing with timers and interrupts)

  DigitalWriteFast library: https://github.com/watterott/Arduino-Libs
  Copyright (c) 2011, 2012, 2013, 2014, 2015 Watterott electronic (www.watterott.com)

********************************************************************************************************

   2025.05.05
   V02.15   Zünkurve aug 6 Punkte reduziert, ab 5000Upm 37° Vorzündung; DelayPerRot von int8_t nach uint8_t geändert (wie es original mal war)
   V02.14   calcDelay() überflüssiges gelöscht; settings.h Zündkurve angepasst bis 9900Upm
   V02.13   Watchdog hinzugfügt
   CX500 Arduino CDI by Brummbaehr
   Test mit ComVisuV1.1 https://www.mikrocontroller.net/topic/388750?page=1

   Eingänge:
   D2 = PicUp 1 = rechter Zylinder
   D3 = PicUp 2 = linker Zylinder
   D8 = Neutralschalter
   D9 = Seitenständerschalter
   D12 = USB Power ON, Ausgabe auf seriellen Monitor
   A0 = Messung der Batteriespannung über Spannungsteiler 82K zu 18K

   Ausgänge:
   D4 = Zündspule 1 = rechter Zylinder
   D5 = Zündspule 2 = linker Zylinder
   D6 = Ausgang Drehzahlmesser
   D7 = DC/DC Wandler abschalten
   D13 = Status LED

   *********************************************************************
   ******************* beide 15° Impulsgeber defekt ********************
   *********************************************************************
   * kleinste Drehzahl zum Start = 150 Upm
   * 
   * Die Timerzeit wird anhand der Umdrehungszeit für BASE_ADVANCE - 10° (also 10° Vorzündung) berechnet
   * Wenn dann der 15° Impulsgeber kommt, dann sofort zünden und den Timer beenden
    
*/

#include <digitalWriteFast.h> // library for fast pin read and write
#include <stdint.h>
#include "honda_cx500_settings.h"
#include <avr/wdt.h>

#define PICUP_PIN1 2 // Picup rechter Zylinder
#define PICUP_PIN2 3 // Picup linker Zylinder
#define COIL1 4 // Ausgang rechte Zündspule
#define COIL2 5 // Ausgang linke Zündspule
#define DZMPin 6 // Ausgang
#define DCDC 7 // Ausgang DC/DC Wandler 0 => Abgeschaltet
#define NEUTRALPin 8 // Eingang
#define STAENDERPin 9 // Eingang
#define LEDPin 13 //
#define USBPin 12 // Eingang für USB Erkennung !!Pulldown!!
#define DTRPin 11 // Eingang für DTR Pin vom USB-Chip (02.10.2023 V206)
#define BATTERIEPin A0 // Analog Eingang für Batteriespannung
#define SCHALTER false // Seitenständerschalter aktivieren wenn true
#define DZMIMPULSE 1 // Anzahl der Impulse pro Umdrehung

uint16_t VERSION = 215;

volatile int32_t coilDelayTime = 15000;
const int32_t OUTPUT_INTERVAL = 1000000; //=1 sec, Time to check if engine runs
const uint8_t NO_OF_SIGNALS_FOR_RPM_CALCULATION = 4; // rpm is averaged over the number of rotations defined here
const float MAXBATTERIE = 18.0; // Bei Überspannung Zündung abschalten

const byte TIMER_DELAY_FACTOR = 4;  // microseconds, correction factor for timer
int16_t signals = 0; // no. of hall signals since last rpm calculation
//uint32_t sumTime2 = 0; // sum of time difference of signals at 2nd coil (used for RPM calc.)
int32_t rpm = 0;

uint32_t timerInterval = 15000; //defaultwert, sonst wird der DCDC beim ersten Zündimpuls zu schnell wieder eingeschaltet
bool timerOn = false;
bool engineStopped = true; // flag will be set if no interrupt from hall sensor is received for longer than MAX_INACTIVE_DURATION µs
bool start = false; // Drehzahl ist kleiner als der erste Wert in der Tabelle
// volatile bool DCDC_ON = true;
bool USB_on = false;
bool Stop = false;
bool Over_voltage = false;
bool Ausgabe = false;
volatile bool Picup1akt = false;
volatile bool Picup1_37 = false;
volatile bool Picup1_15 = false;
volatile bool Picup2akt = false;
volatile bool Picup2_37 = false;
volatile bool Picup2_15 = false;
volatile bool allowLoadCoil; // usually true, only false if engine is too fast or to low

//volatile int32_t newTriggerTime;
//volatile int32_t oldTriggerTime;

uint32_t newTriggerTime1 = 0;
uint32_t oldTriggerTime1 = 0;
uint32_t newTriggerTime2 = 0;
uint32_t oldTriggerTime2 = 0;
uint32_t Timerstoptime = 0;
volatile int32_t isr1 = 0;
volatile int32_t isr2 = 0;
uint32_t currTime = 0;
uint32_t DCDC_ON_Time = 0;
uint32_t lastimpuls = 0;
float Batteriespannung = 0;
byte outcase;

int32_t usRotationEngine[ TABLE_SIZE ]; // table containing the rotation times of the engine (in µs)
int32_t usDelay[ TABLE_SIZE ]; // the corresponding delay times (in µs) at the engine
uint8_t DelayPerRot[ TABLE_SIZE ]; // slope between two advanced tab pairs (range is about 90 to 130, for calcualtiosn divided by 128 to get "float" values
const int32_t RPM2MICROS = 60000000; // Umrechnungsfaktor (60s in µS) für eine Umdrehung in RPM
uint8_t curveIndex = 0; // index of currently used ignition curve
int32_t usHallDifference; // the current time in µs for last rotation

uint8_t currentIndex = 255; // current index for tablular data
int32_t usCurrentOneRotationEngine = 0; // the current time in µs for a rotation of the crankshaft


void setup() {
  //pinMode (A1, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A2, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A3, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A4, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A5, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A6, OUTPUT); //offene Pins als OUTPUT
  //pinMode (A7, OUTPUT); //offene Pins als OUTPUT
  pinMode (PICUP_PIN1, INPUT_PULLUP);
  pinMode (PICUP_PIN2, INPUT_PULLUP);
  pinMode (USBPin, INPUT);
  pinMode (COIL1, OUTPUT);
  pinMode (COIL2, OUTPUT);
  pinMode (DCDC, OUTPUT);
  pinMode (LEDPin, OUTPUT);
  pinMode (DZMPin, OUTPUT);
  pinMode (NEUTRALPin, INPUT_PULLUP);
  pinMode (STAENDERPin, INPUT_PULLUP);
  pinMode (USBPin, INPUT);  // Kein Pullup!!
  pinMode (DTRPin, INPUT_PULLUP);  // Kein Pullup!! V206
  digitalWriteFast (DCDC, LOW);   //V2.13 DCDC Wandler beim Boot erst mal abschalten
  Serial.begin(115200);
  USB_on = digitalRead(USBPin) || !digitalRead(DTRPin) ; //(02.10.2023 V206)
  calculateArrayValues();
  Reset();
  wdt_enable(WDTO_2S); // 20.02.2025 Watchdog aktivieren 2s V212
}


// ------------------------------------------------------------------------------
// Procedure to activate interrupt for hall sensor
// ------------------------------------------------------------------------------
void activateInterrupt ()
{
  attachInterrupt(digitalPinToInterrupt( PICUP_PIN1 ), ISR_Picup1, FALLING );
  attachInterrupt(digitalPinToInterrupt( PICUP_PIN2 ), ISR_Picup2, FALLING );
}  // end of activateInterrupt

// ------------------------------------------------------------------------------
// Procedure to deactivate interrupt for hall sensor
// ------------------------------------------------------------------------------
void deactivateInterrupt ()
{ Serial.println( "deactivateInterrupt" );
  detachInterrupt( digitalPinToInterrupt( PICUP_PIN1 ) );
  detachInterrupt( digitalPinToInterrupt( PICUP_PIN2 ) );
}  // end of deactivateInterrupt

void ISR_Picup1()
{ 
  Picup1akt = true; 
  isr1++;
  digitalWriteFast (DCDC, LOW); //DC/DC-Wandler abschalten (V1.62)
  if (Picup2akt) 
  {
    Picup1_37 = true;
    Picup2akt = false;
    oldTriggerTime1 = newTriggerTime1;
    newTriggerTime1 = micros();

//    newTriggerTime = newTriggerTime1;   //Zeiten für calcDelay()
//    oldTriggerTime = oldTriggerTime1;   //Zeiten für calcDelay()
    
    allowLoadCoil = calcDelay(); // coilDelayTime berechnen; Ist 0 wenn Drehzahl zu hoch
    timerInterval = (uint32_t( coilDelayTime - TIMER_DELAY_FACTOR )) * 2 - 1;
    if (start) timerInterval = ((newTriggerTime1 - oldTriggerTime1) / 180) * (BASE_ADVANCE - 10); // Umdrehungszeit / 360 = Zeit für 1°; Das mal 32° für 10° Vorzündung (alles *2)
    
    startTimer();
              
  }else {
         Picup1_15 = true;
         if (start) { // Wenn der zweite Impuls kommt, dann beim Starten zünden und Timer stoppen
                    digitalWrite (COIL1, HIGH);
                    stopTimer();
                    start = false; //V200
                    }
        }
  engineStopped = false;
}

void ISR_Picup2()
{ 
  Picup2akt = true;
  isr2++;
  digitalWriteFast (DCDC, LOW);  //DC/DC-Wandler abschalten (V1.62)
  if (Picup1akt) 
  {
    Picup2_37 = true;
    Picup1akt = false;
    oldTriggerTime2 = newTriggerTime2;
    newTriggerTime2 = micros();
    
//    newTriggerTime = newTriggerTime2;   //Zeiten für calcDelay()
//    oldTriggerTime = oldTriggerTime2;   //Zeiten für calcDelay()
    
    allowLoadCoil = calcDelay(); // coilDelayTime berechnen; Ist 0 wenn Drehzahl zu hoch
    timerInterval = (uint32_t( coilDelayTime - TIMER_DELAY_FACTOR )) * 2 - 1;
    if (start) timerInterval = ((newTriggerTime2 - oldTriggerTime2) / 180) * (BASE_ADVANCE - 10); // Umdrehungszeit / 360 = Zeit für 1°; Das mal 32° für 10° Vorzündung (alles *2)
               
    startTimer();
    }else {
          Picup2_15 = true;
          if (start) {  // Wenn der zweite Impuls kommt, dann beim Starten zünden und Timer stoppen
                      digitalWrite (COIL2, HIGH);
                      stopTimer();
                      start = false; //V200
                     }
          }
  engineStopped = false;
}

// ------------------------------------------------------------------------------
// interrupt for when time to turn coil on
// ------------------------------------------------------------------------------
ISR (TIMER1_COMPA_vect)
{ // Zündfunken auslösen, wenn Timer abgelaufen
  //digitalWriteFast (DCDC, LOW); //DC/DC-Wandler abschalten
  //DCDC_ON = false;
  if (allowLoadCoil) // ist 0, wenn Drehzahl zu hoch oder niedrig
  {
    if (Picup1akt) {
      digitalWriteFast (COIL1, HIGH);
      //timer1--;
      //Serial.println( "Picup1_Timer " );
    }

    if (Picup2akt) {
      digitalWriteFast (COIL2, HIGH);
      //timer2--;
      //Serial.println( "Picup2_Timer " );
      start = false; //V210 Dann kann der DC/DC Wandler nach Pausenzeit wieder aktiviert werden, auch wenn der Picup2_15 nicht kommt!
    }
  }
  stopTimer();

}  // end of TIMER1_COMPA_vect


// ------------------------------------------------------------------------------
// Procedure to start timer
// ------------------------------------------------------------------------------
void startTimer()
{
  TCCR1A = 0;                       // normal mode
  TCCR1B = 0;                         // stop timer
  TCNT1 = 0;                          // count back to zero
  TCCR1B = bit(WGM12) | bit(CS11);    // CTC, scale to clock / 8
  if ( timerInterval > 65535 )
  {
    OCR1A = 65535;
  }
  else
  {
    OCR1A = timerInterval;
  }
  TIMSK1 = bit (OCIE1A);
  timerOn = true;
  //timer++;
}  // end of startTimer


// ------------------------------------------------------------------------------
// Procedure to stop timer
// ------------------------------------------------------------------------------
void stopTimer()
{
  TCCR1B = 0;                         // stop timer
  TCNT1 = 0;                          // count back to zero
  TIMSK1 = 0;                         // cancel timer interrupt
  timerOn = false;
  Timerstoptime = micros();
  //digitalWrite (COIL1, LOW);
  //digitalWrite (COIL2, LOW);
  //digitalWrite (DCDC, HIGH);
  //  timer--;
}  // end of stopTimer



void calculateArrayValues()
{
  /*
      Hier wird nun eine Tabelle aus den Daten der Vorzündungskurve berechnet
      (siehe honda_cx500_settings.h)
      int16_t usRotationEngine[ TABLE_SIZE ]; // table containing the rotation times of the engine (in µs)
      int16_t usDelay[ TABLE_SIZE ]; // the corresponding delay times (in µs) at the engine
      uint8_t DelayPerRot[ TABLE_SIZE ]; // slope between two advanced tab pairs (range is about 90 to 130, for calcualtiosn divided by 128 to get "float" values
  */

  // define variables
  uint8_t dataIndex;
  float adv1, adv2;
  float rpm1, rpm2, rpm_min, rpm_max;
  float currRPM, curr_USROT, advance;
  float slope, intercept;
  float usPerDegreeEngine, tmp;

  // init variables
  dataIndex = curveIndex * NO_OF_POINTS_PER_CURVE;
  curr_USROT = MAX_USROT_ENG; // start with max time = min RPM
  adv1 = float( CURVE_DATA[ dataIndex ] [ 1 ] ); // 1st advance for selected curve
  adv2 = float( CURVE_DATA[ dataIndex + 1 ] [ 1 ] ); // 2nd advance for selected curve
  rpm1 = 100 * float( CURVE_DATA[ dataIndex ] [ 0 ] ); // 1st rpm for selected curve, multiplied by 100 (see above), in crank system
  rpm2 = 100 * float( CURVE_DATA[ dataIndex + 1 ] [ 0 ] ); // 2nd rpm for selected curve, multiplied by 100 (see above), in crank system
  rpm_min = 100 * float( CURVE_DATA[ dataIndex ] [ 0 ] );
  rpm_max = 100 * float( CURVE_DATA[ dataIndex + NO_OF_POINTS_PER_CURVE - 1 ] [ 0 ] );
  slope = ( adv2 - adv1 ) / ( rpm2 - rpm1 );
  intercept = adv1 - rpm1 * slope; //intercept for current line segment
  /*        Serial.print( "slope=" );
          Serial.print( slope, 4 );
          Serial.print( " intercept=" );
          Serial.println( intercept, 1 ); */
  if (USB_on)
  {
    Serial.println( "Calculate array values");
  }


  for (int i = 0; i <= MAX_TABLE_INDEX; i++)
  {
    currRPM = float( RPM2MICROS ) / curr_USROT;
    usPerDegreeEngine = curr_USROT / 360;
    if (USB_on) {
      Serial.print( "currRPM=" );
      Serial.println( currRPM, 0 );
      Serial.print( "rpm1=" );
      Serial.println( rpm1 );
      Serial.print( "rpm2=" );
      Serial.println( rpm2 );
      Serial.print( "rpm_min=" );
      Serial.println( rpm_min );
      Serial.print( "rpm_max=" );
      Serial.println( rpm_max );
      Serial.print( "dataIndex=" );
      Serial.println( dataIndex );
    }
    if ( currRPM <= rpm_min ) // below 1st rpm in curve data, machine idles
      advance = adv1;
    else if ( currRPM >= rpm_max )  // above last rpm in curve data
      advance = adv2;
    else // intermediate
    { if ( currRPM > rpm2 ) // get next interval
      {
        dataIndex++;

        // Get data for current curve interval by linear interpolation
        adv1 = float( CURVE_DATA[ dataIndex ] [ 1 ] ); // next advance for selected curve
        adv2 = float( CURVE_DATA[ dataIndex + 1 ] [ 1 ] ); // next+1 advance for selected curve
        rpm1 = 100 * float( CURVE_DATA[ dataIndex ] [ 0 ] ); // next rpm for selected curve
        rpm2 = 100 * float( CURVE_DATA[ dataIndex + 1 ] [ 0 ] ); // next+1 rpm for selected curve
        slope = ( adv2 - adv1 ) / ( rpm2 - rpm1 );
        intercept = adv1 - rpm1 * slope;
        if (USB_on) {
          Serial.print( "slope=" );
          Serial.print( slope, 4 );
          Serial.print( " intercept=" );
          Serial.println( intercept, 1 );
        }
      }
      advance = currRPM * slope + intercept; // use linear interpolation from the curve data
      if (USB_on)
        Serial.println( "advance=" + String(advance) );
    }
    // calculate the time we have to wait after the sensor has fired
    // Remember: In the loop curr_USROT is in descending order, in the array we need the data in ascending order!
    usDelay[MAX_TABLE_INDEX - i] = int32_t( ( float( BASE_ADVANCE ) - advance) * usPerDegreeEngine );
    usRotationEngine[MAX_TABLE_INDEX - i] = int32_t( curr_USROT );

    curr_USROT -= USROT_INCR_ENG; //decrement the µs for the next iteration
  }

  // calculate slope values (as uint8_t data)
  uint8_t j;
  for (uint8_t i = 0; i < MAX_TABLE_INDEX; i++)
  {
    j = i + 1;
    tmp = float( usDelay[j] - usDelay[i] ) / float( usRotationEngine[j] - usRotationEngine[i] );
    DelayPerRot[i] = uint8_t( tmp * 128 ); // *128 to bring value to uint8_t range
    if (USB_on) {
      Serial.println( "i=" + String(i) );
      Serial.println( "usDelay[j]=" + String(usDelay[j]) );
      Serial.println( "usDelay[i]=" + String(usDelay[i]) );
      Serial.println( "usDelay[j] - usDelay[i]=" + String(usDelay[j] - usDelay[i]) );
      Serial.println( "usRotationEngine[j] - usRotationEngine[i]=" + String(usRotationEngine[j] - usRotationEngine[i]) );
      Serial.println( "tmp=" + String(tmp) );
      Serial.println( "DelayPerRot=" + String(DelayPerRot[i]) );
    }
  }
  DelayPerRot[MAX_TABLE_INDEX] = DelayPerRot[MAX_TABLE_INDEX - 1];

  // Output for test purposes in correct sequence
  if (USB_on)
  {
    Serial.println( "Curve index=" + String(curveIndex) );
    Serial.println( "TABLE_SIZE=" + String(TABLE_SIZE) );
    Serial.println( "Max table index=" + String(MAX_TABLE_INDEX) );
    Serial.println( "i;RPM;USROTationEng;usDelay;Slope");
    curr_USROT = MIN_USROT_ENG;
    for (uint8_t i = 0; i <= MAX_TABLE_INDEX; i++)
    {
      currRPM = float( RPM2MICROS ) / curr_USROT;
      Serial.println( String(i) + ";" + String(currRPM) + ";" + String( usRotationEngine[i]) + ";" + usDelay[i] + ";" + String( DelayPerRot[i] ));
      curr_USROT += USROT_INCR_ENG; //increment the µs for the next iteration
    }
  }
} // end of calculateArrayValues()

// ------------------------------------------------------------------------------
// procedure to calculate the current delay value (depending on current rotation time)
// ------------------------------------------------------------------------------
bool calcDelay()
/* Example data for my Honda
   Remember: usRotationEngine is ordered in ascending order (decreasing rpm)
  Curve index=0
  Max table index=74
  i;RPM;usHalfRotationEng;usDelay;Slope*128
  0;30000.00;1000;822;105
*/

{
  bool success;
  int32_t usCoilDelayOffset;
  int32_t usCoilDelay;

  /*
  int32_t newTriggerTime;
  int32_t oldTriggerTime;

  if (Picup1akt)
  {
  newTriggerTime = newTriggerTime1;
  oldTriggerTime = oldTriggerTime1;
  }
  
  if (Picup2akt)
  {
  newTriggerTime = newTriggerTime2;
  oldTriggerTime = oldTriggerTime2;
  }  */

  // This number will overflow (go back to zero), after approximately 70 minutes
  // if overflow: usHalfRotationCam is not changed
  if ( newTriggerTime1 > oldTriggerTime1 )
  {
    usHallDifference = (newTriggerTime1 - oldTriggerTime1); //
    usCurrentOneRotationEngine = usHallDifference;
  }

  // Check engine state
  if (usCurrentOneRotationEngine < MIN_ALLOWED_USROT_ENG )  // engine running much too fast (beyond allowed range)
  {
    currentIndex = 0;
    success = false;
  }
  else if (usCurrentOneRotationEngine > MAX_ALLOWED_USROT_ENG )  // engine running much too slow (below allowed range)
  {
    currentIndex = MAX_TABLE_INDEX;
    success = false;
  }
  else
  {
    if ( usCurrentOneRotationEngine > usRotationEngine[MAX_TABLE_INDEX] ) // engine running slow (outside tabular values), but in allowed range
    {
      currentIndex = MAX_TABLE_INDEX;
      start = true;
    }
    else if ( usCurrentOneRotationEngine < usRotationEngine[0] ) // engine running fast (outside tabular values), but in allowed range
    {
      currentIndex = 0;
      start = false;
    }
    // V214 angepasst, weil überflüssig
    else 
    {
      currentIndex = BinarySearchIndex();
      start = false;
    }

    /*
    else if ( usCurrentOneRotationEngine < usRotationEngine[ currentIndex ] )  // engine speed within tabular values, increasing (time decreases!)
    {
      currentIndex = BinarySearchIndex();
      start = false;
    }
    else if ( usCurrentOneRotationEngine > ( usRotationEngine[ currentIndex  ] + USROT_INCR_ENG )) // engine speed within tabular values, decreasing(time increases!)
    {
      currentIndex = BinarySearchIndex(); // + 1;  Die 1 führt zu ganzen zündaussetzer // when engine speed is decreasing, usCurrentOneRotationEngine (which has a time lag) maybe too high
      start = false;                                        //for safety reasons, we use the next (slower) computed interval
      if ( currentIndex > MAX_TABLE_INDEX )
      {
        currentIndex = MAX_TABLE_INDEX;
      }
    }*/
    success = true;
  }

  if ( success && ( currentIndex == MAX_TABLE_INDEX ) ) // Engine speed ok but slower than slowest tabular value
  {
    usCoilDelay = usDelay[currentIndex];// usCurrentOneRotationEngine - (usCurrentOneRotationEngine>>5); // correct
    //Serial.println( usCoilDelay );
    coilDelayTime = usCoilDelay;
  }
  else if ( success )
  {
    // if table value found, use linear interpolation (slope is stored in tabular values, but multiplied by 128) in current interval: y = yi + a*(x - xi)
    // >>7 means division by 128 to get correct slope
    usCoilDelayOffset = ( int32_t( DelayPerRot[ currentIndex ] ) * ( int32_t( usCurrentOneRotationEngine - usRotationEngine[ currentIndex  ] ) ) ) >> 7;
    usCoilDelay = int32_t( usDelay[ currentIndex ] ) + usCoilDelayOffset;

    if ( usCoilDelay > usCurrentOneRotationEngine ) // special case just in case
    {
      //currentDwellTime = DWELL_TIME_NORMAL;
      usCoilDelay = usCurrentOneRotationEngine - 100;
    }
    else
    {
      ;
    }
    // Now Correct for coil dwell time
    coilDelayTime = usCoilDelay;
  }
  //  if (!success)
  //  Serial.println( !success );

  return success;
} // end calcDelay




// ------------------------------------------------------------------------------
// Binary search algorith "How to search an ordered table" from "Numerical recipies"
// ------------------------------------------------------------------------------
byte BinarySearchIndex()
{
  uint8_t ju, jm, jl;
  jl = 0;
  ju = MAX_TABLE_INDEX;

  while ( ju - jl > 1 )
  {
    jm = ( ju + jl ) >> 1;
    if ( usCurrentOneRotationEngine >= usRotationEngine[ jm ] )
      jl = jm;
    else
      ju = jm;
  }
  return jl;
}


// ------------------------------------------------------------------------------
// procedure reset all variables to defaults
// ------------------------------------------------------------------------------
void Reset()
{
  // Initialize timer
  TCCR1A = 0;                       // normal mode
  TCCR1B = 0;                       // stop timer
  TCNT1 = 0;                        // count back to zero
  TIMSK1 = 0;                       // cancel timer interrupt
  TCCR1B = bit(WGM12) | bit(CS11);  // CTC, scale to clock / 8

  /*
    oldTriggerTime1 = 0; //
    oldTriggerTime2 = 0; //
    newTriggerTime1 = 0; //
    newTriggerTime2 = 0; //
  */
  digitalWriteFast (COIL1, LOW);
  digitalWriteFast (COIL2, LOW);
  allowLoadCoil = false;  // V210 von true nach false, damit die LED nicht kurz an geht
  usCurrentOneRotationEngine = 0;
  currentIndex = MAX_TABLE_INDEX; // default for "unknown"
  signals = 0;
  activateInterrupt();
  if (USB_on)
    Serial.println( "RESET");
}


// ------------------------------------------------------------------------------
// procedure to check if engine is still running (hall sensor has recently fired)
// ------------------------------------------------------------------------------
void checkEngineRunning()
{
  // Check if engine is still running
  if ( !engineStopped && (currTime - max(oldTriggerTime1, oldTriggerTime2)) > MAX_INACTIVE_DURATION )
  {
    engineStopped = true;
    if (USB_on) {
      Serial.print( "#9F" );
      Serial.print(isr1);
      Serial.print( ";" );

      Serial.print( "#10F" );
      Serial.print(isr2);
      Serial.print( ";" );

      /*
      Serial.println( "Engine stopped" );
      Serial.print( "isr1: " );
      Serial.println( isr1 );
      Serial.print( "isr2: " );
      Serial.println( isr2 );
      Serial.print( "currTime: " );
      Serial.println( currTime );
      Serial.print( "max: " );
      Serial.println( max(oldTriggerTime1, oldTriggerTime2) );
      Serial.print( "currTime - max: " );
      Serial.println( currTime - max(oldTriggerTime1, oldTriggerTime2) );*/
    }
    deactivateInterrupt();
    Reset();
  }
} // end checkEngine


void Monitor() {
  static float advance;  
  Ausgabe = false;    //serielle Ausgabe wieder sperren
  
  if ( signals >= NO_OF_SIGNALS_FOR_RPM_CALCULATION ) {

        
        rpm = int32_t( float( RPM2MICROS ) / ( newTriggerTime1 - oldTriggerTime1) );//calcRPM( sumTime2 );
        Serial.print( "#1F" );
        Serial.print(rpm);
        Serial.print( ";" );

        if ( usCurrentOneRotationEngine > usRotationEngine[MAX_TABLE_INDEX] ) // engine running slow (outside tabular values), but in allowed range
        //if (rpm > 750)                                                      // unter 750 UPM ist coilDelayTime ungültig. Da wird beim 2ten Impuls gezündet
        advance = CURVE_DATA[ 0 ] [ 1 ];
        else
        advance = ( float(BASE_ADVANCE) - (float(coilDelayTime) - TIMER_DELAY_FACTOR) / (float(usHallDifference) / 360) );
        
        Serial.print( "#2F" );
        Serial.print(advance);
        Serial.print( ";" );
        
        signals = 0;
        outcase--;    //ansonsten wird eine Ausgabe übersprungen
        return;
  }

    if (engineStopped) {
    rpm = 0;
    Serial.print( "#1F" );
    Serial.print(rpm);
    Serial.print( ";" );
    advance = CURVE_DATA[ 0 ] [ 1 ];
    Serial.print( "#2F" );
    Serial.print(advance);
    Serial.print( ";" );
    //sumTime2 = 0;
    signals = 0;
  }

  switch (outcase) {

    case 1:
      Serial.print( "#3F" );
      Serial.print(Stop);
      Serial.print( ";" );
      break;

    case 2:
      Serial.print( "#4F" );
      Serial.print(digitalReadFast(NEUTRALPin));
      Serial.print( ";" );
      //outcase = 0;
      break;

    case 3:
      Serial.print( "#5F" );
      Serial.print(digitalReadFast(STAENDERPin));
      Serial.print( ";" );
      //outcase = 0;
      break;

    case 4:
      Serial.print( "#6F" );
      if (engineStopped) // Batteriespannung in jedem Zyklus einlesen wenn Motor steht
        Batterie();
      Serial.print(Batteriespannung);
      Serial.print( ";" );
      //outcase = 0;
      break;

    case 5:
      Serial.print( "#7F" );
      Serial.print(Over_voltage);
      Serial.print( ";" );
        if (!engineStopped)  // Wenn der Motor läuft, dann keine Ausgabe von VERSION
        ; //outcase = 0;
      break;

    case 6:               // Nur wenn Motor nicht läuft
      Serial.print( "#8F" );
      Serial.print(VERSION);
      Serial.print( ";" );
      //outcase = 0;
      break;

    case 7:               // Nur wenn Motor nicht läuft
      Serial.print( "#9F" );
      Serial.print(isr1);
      Serial.print( ";" );
      //outcase = 0;
      break;

    case 8:               // Nur wenn Motor nicht läuft
      Serial.print( "#10F" );
      Serial.print(isr2);
      Serial.print( ";" );
      outcase = 0;
      break;
  }
}

void Drehzahlimpuls()
{
  if (engineStopped)
  {
    digitalWriteFast (DZMPin, LOW);
    lastimpuls = currTime;
  } else {
    if ( lastimpuls - 128 + usHallDifference / (2 * DZMIMPULSE) < currTime ) // Die 128 ist zur Korrektur anhand der Messungen mit dem Ozzi
    {
      digitalWriteFast (DZMPin, !digitalReadFast(DZMPin));
      lastimpuls = currTime;
    }
  }
}

void input()
{
  static int32_t lastTime;
  if (!digitalReadFast(NEUTRALPin) || !digitalReadFast(STAENDERPin)) {
    Stop = false;
    lastTime = currTime;
  }
  if (lastTime + 200000 < currTime) // nach 200ms Verzögerung Stop
    Stop = SCHALTER; // Nur wenn SCHALTER = true in der Deklaration
}

void Batterie()
{
  int spannung = analogRead(BATTERIEPin); // Dauert 100us!
  Batteriespannung = map(spannung, 0, 1023, 0.00, 500 * 5.55) / 100.00;
  if (Batteriespannung > MAXBATTERIE)
    Over_voltage = true;
}

// ***************************************************************************************
// **** LOOP
// ***************************************************************************************
void loop()
{
  currTime = micros();
  
if (Picup1_37 && !timerOn && Timerstoptime + timerInterval < currTime)  // Nur wenn der Timer schon gefeuert hat **V210 15° Picup ggf. defekt
  {
    digitalWriteFast (COIL1, LOW);
    Picup1_37 = false;
  }

if (Picup2_37 && !timerOn && Timerstoptime + timerInterval < currTime)  // Nur wenn der Timer schon gefeuert hat **V210 15° Picup ggf. defekt
  {
    digitalWriteFast (COIL2, LOW);
    Picup2_37 = false;
    Ausgabe = true;     //serielle Ausgabe freigeben
    signals++; // increment signal count
  }

  //DCDC_Wandler alle 500ms mal kurz ausschalten, sollte ein Thyristor (warum auch immer) gezündet haben
if (engineStopped) 
{
  oldTriggerTime1 = oldTriggerTime2 = newTriggerTime1 = newTriggerTime2 = currTime; // Alle Zeiten auf aktuelle Zeit
  digitalWriteFast (COIL1, LOW);
  digitalWriteFast (COIL2, LOW);
  
  if (digitalReadFast (DCDC) && DCDC_ON_Time + 500000 < currTime) //Alle 500ms ausschalten
  {
    digitalWriteFast (DCDC, LOW); //DC/DC-Wandler ausschalten
    DCDC_ON_Time = currTime;
  }
  if (!digitalReadFast (DCDC) && DCDC_ON_Time + 1000 < currTime) // nach 1ms wieder einschalten
  {
    digitalWriteFast (DCDC, !Over_voltage && !Stop); //DC/DC-Wandler einschalten
    DCDC_ON_Time = currTime;
    digitalWriteFast (LEDPin, !digitalReadFast (LEDPin) && !Over_voltage && !Stop);       // LED blinkt
  }
} else {
  digitalWriteFast (LEDPin, allowLoadCoil && !Over_voltage && !Stop);  // LED einschalten, solange Drehzahl im gültigen Bereich und keine Abschaltung
  
// ***********   DCDC Wandler einschalten ************
    if (timerOn || digitalReadFast(COIL1) || digitalReadFast(COIL2) || start)    //Wenn ein Timer aktiv ist, oder ein Tyristor angesteuert ist,
        DCDC_ON_Time = currTime;                                        //dann die Zeit aktualisieren, so das der DCDC Wandler NICHT! aktiviert wird
    if (!timerOn && DCDC_ON_Time + timerInterval < currTime)            //Zusätzliche Verzögerung nach dem Ausschalten der Tyristoren  
        {                                                               //Wenn die Zeit erreicht ist, dann erst den DCDC Wandler wieder einschalten
          digitalWriteFast (DCDC, !Over_voltage && !Stop);              //
        } else digitalWriteFast (DCDC, LOW);                            //DCDC Wandler ausschalten wenn dem nicht so ist. Wichtig!! 
      }                                                                 //Der ISR kann zu jeder Zeit das Programm unterbrechen, also auch genau vor dem Wiedereinschalten des DCDC

if ( USB_on && ((Ausgabe && digitalReadFast (DCDC) && !timerOn) || engineStopped)) // Serielle Ausgaben, wenn USB-Kabel angesteckt und Zündvorgang für aktuelle KW-Umdrehung beendet, oder Motor steht
{ //Serial.println( isr1 - isr2 ); Ausgabe = false;
  outcase++;
  Monitor();
}

if ( currTime - max(newTriggerTime1, newTriggerTime2) >= OUTPUT_INTERVAL ) { // Wenn 1s keine neuen Signale gekommen sind, dann steht vermutlich der Motor
  checkEngineRunning(); // Is engine still running ?
  }

if ( digitalReadFast (DCDC) && !timerOn ) // Wenn DCDC-Wandler aktiv ist und kein Timer aktiv ist, dann ist Zeit für die Batteriespannung zu messen
  Batterie(); // Batteriespannung checken
    
Drehzahlimpuls();
input();
wdt_reset();
};
